#include <vector>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <map>
#include <cassert>
#include <stdexcept>
#include <chrono>
#include <format>
#include <stdint.h>


static const double GeV = 1000;


//************************************************************************
// Very simple random number generator.


/// Generate a floating-point random number between @c rmin and @c rmax.
float randf (float rmax, float rmin = 0)
{
  static constexpr uint32_t rngmax = static_cast<uint32_t> (-1);
  static uint32_t seed = 1;
  seed = (1664525*seed + 1013904223);
  return static_cast<float>(seed) / static_cast<float>(rngmax) * (rmax-rmin) + rmin;
}


/// Generate an integer random number between @c rmin and @c rmax.
int randi (int rmax, int rmin = 0)
{
  return static_cast<int> (randf (rmax, rmin));
}


//************************************************************************
// Timer class.


class Timer
{
public:
  using clock_t = std::chrono::steady_clock;
  using time_point_t = std::chrono::time_point<clock_t>;
  using duration_t = std::chrono::duration<float>;

  void start (const char* name);
  void stop (const char* name);
  void dump();

private:
  struct Timeent {
    duration_t elapsed;
    time_point_t starttime;
    bool running = false;
  };

  std::map<std::string, Timeent> m_timers;
};


void Timer::start (const char* name)
{
  Timeent& ent = m_timers[name];
  assert (!ent.running);
  ent.starttime = clock_t::now();
  ent.running = true;
}


void Timer::stop (const char* name)
{
  Timeent& ent = m_timers[name];
  assert (ent.running);
  ent.elapsed += clock_t::now() - ent.starttime;
  ent.running = false;
}


void Timer::dump()
{
  for (const auto& x : m_timers) {
    std::cout << std::format ("{:5} {:.02f}\n", x.first, x.second.elapsed.count());
  }
}


Timer timers;


class TimerGuard
{
public:
  TimerGuard (const char* name);
  ~TimerGuard();

private:
  const char* m_name;
};


TimerGuard::TimerGuard (const char* name)
  : m_name (name)
{
  timers.start (m_name);
}


TimerGuard::~TimerGuard()
{
  timers.stop (m_name);
}


//************************************************************************


class Fourvec
{
public:
  Fourvec (double x = 0, double y = 0, double z = 0);
  Fourvec (const Fourvec& other);
  Fourvec& operator= (const Fourvec&) = default;
  virtual ~Fourvec();
  double x() const { return m_x; }
  double y() const { return m_y; }
  double z() const { return m_z; }
  double e() const { return m_e; }

  Fourvec& operator+= (const Fourvec& other);


  static int s_count;

private:
  double m_x;
  double m_y;
  double m_z;
  double m_e;
};


int Fourvec::s_count = 0;



Fourvec::Fourvec (double x, double y, double z)
  : m_x (x), m_y (x), m_z (z), m_e (std::sqrt (x*x + y*y + z*z))
{
  ++s_count;
}


Fourvec::Fourvec (const Fourvec& other)
{
  ++s_count;
  m_x = other.m_x;
  m_y = other.m_y;
  m_z = other.m_z;
  m_e = other.m_e;
}


Fourvec::~Fourvec() { --s_count; }

Fourvec& Fourvec::operator+= (const Fourvec& other)
{
  m_x += other.m_x;
  m_y += other.m_y;
  m_z += other.m_z;
  m_e += other.m_e;
  return *this;
}


typedef Fourvec Track;


class Cluster
  : public Fourvec
{
public:
  Cluster (const Fourvec& v,
           std::unique_ptr<Track> track = std::unique_ptr<Track>());

  Cluster(const Cluster& other);
  Cluster(Cluster&& other) = default;
  Cluster& operator= (const Cluster& other);


private:
  std::unique_ptr<Track> m_track;
};



Cluster::Cluster (const Fourvec& v, std::unique_ptr<Track> track)
  : Fourvec (v),
    m_track (std::move (track))
{
}


Cluster::Cluster(const Cluster& other)
  : Fourvec(other)
{
  if (other.m_track)
    m_track = std::make_unique<Track> (*other.m_track);
}


Cluster& Cluster::operator= (const Cluster& other)
{
  if (this != &other) {
    Fourvec::operator= (other);
    if (other.m_track)
      m_track = std::make_unique<Track> (*other.m_track);
    else
      m_track.reset();
  }
  return *this;
}


typedef std::vector<Cluster> ClusterVec;


//************************************************************************


Fourvec randvec()
{
  return Fourvec (randf(100*GeV,-100*GeV),
                  randf(100*GeV,-100*GeV),
                  randf(100*GeV,-100*GeV));
}


Cluster makeCluster()
{
  std::unique_ptr<Track> track;
  Fourvec v = randvec();
  if (randi(2) == 0)
    track = std::make_unique<Track> (v.x() * 0.9, v.y() * 0.9, v.z() * 0.9);

  if (randi(1000000) == 0)
    throw std::runtime_error("asd");
  return Cluster (v, std::move(track));
}


ClusterVec makeClusters()
{
  ClusterVec vec;
  vec.reserve (1000);
  for (int i = 0; i < 1000; i++)
    vec.push_back (makeCluster());
  return vec;
}


Fourvec sumClusters (const ClusterVec& vec)
{
  Fourvec sum;
  for (const Cluster& c : vec)
    sum += c;
  return sum;
}


double doit()
{
  TimerGuard timer ("doit");
  ClusterVec vec = makeClusters();
  Fourvec sum = sumClusters (vec);
  return std::hypot (sum.x(), sum.y());
}
 

double doit2()
{
  TimerGuard timer ("doit2");
  Fourvec sum;
  for (int i=0; i < 1000; i++) {
    auto vec = std::make_unique<Cluster> (makeCluster());
    sum += *vec;
  }
  return std::hypot (sum.x(), sum.y());
}
 

int main (int argc, char** argv)
{
  int n = 100000;
  if (argc > 1)
    n = std::atoi (argv[1]);

  double sum1 = 0;
  double sum2 = 0;
  for (int i=0; i < n; i++) {
    try {
      sum1 += doit();
      sum2 += doit2();
    }
    catch (const std::runtime_error&) {
    }
  }

  std::cout << sum1 << " " << sum2 << "\n";
  std::cout << "Objects leaked: " << Fourvec::s_count << "\n";
  timers.dump();
  return 0;
}
